function loadChatForContact(mobile, name) {
    const chatHeader = document.getElementById("chatHeader");
    const messagesList = document.getElementById("messagesList");

    chatHeader.textContent = `${name || 'Lead'} (${mobile})`;
    messagesList.innerHTML = 'Loading...';

    fetch(`../api/getMessages.php?mobile=${encodeURIComponent(mobile)}&page=1`)
        .then(res => res.json())
        .then(data => {
            if (data.status === 'true') {
                messagesList.innerHTML = '';

                data.data.reverse().forEach(msg => {
                    const div = document.createElement("div");
                    div.className = msg.fromPhoneNumber === mobile ? 'message received' : 'message sent';

                    let content = '';
                    switch (msg.messageType) {
                        case 'text':
                            content = msg.text || '[No Text]';
                            break;

                        case 'image':
                            content = `<img src="../api/download.php?id=${msg.mediaId}" alt="Image" style="max-width: 100%; border-radius: 8px;">`;
                            break;

                        case 'video':
                            content = `<video controls style="max-width: 100%; border-radius: 8px;">
                                         <source src="../api/download.php?id=${msg.mediaId}" type="${msg.mimeType}">
                                         Your browser does not support the video tag.
                                       </video>`;
                            break;

                        case 'document':
                            content = `<a href="../api/download.php?id=${msg.mediaId}" target="_blank" download="${msg.fileName || 'Document'}">
                                         📄 ${msg.fileName || 'Download Document'}
                                       </a>`;
                            break;

                        case 'contacts':
                            content = `<div class="contact-card">
                                         <strong>${msg.formattedName || 'Contact'}</strong><br>
                                         <small>${msg.company || ''}</small>
                                       </div>`;
                            break;

                        default:
                            content = `[${msg.messageType} not supported yet]`;
                    }

                    div.innerHTML = `
                        <div class="bubble">${content}</div>
                        <div class="timestamp">${formatIST(msg.timestamp)}</div>
                    `;
                    messagesList.appendChild(div);
                });

                messagesList.scrollTop = messagesList.scrollHeight;
            } else {
                messagesList.innerHTML = 'Failed to load messages.';
            }
        })
        .catch(err => {
            messagesList.innerHTML = 'Error loading messages.';
            console.error(err);
        });
}

function formatIST(utcTimestamp) {
    const date = new Date(utcTimestamp + ' UTC');
    return date.toLocaleString("en-IN", { timeZone: "Asia/Kolkata" });
}


function renderMessages(messages) {
    const container = document.getElementById("messagesList");
    container.innerHTML = '';

    messages.forEach(msg => {
        const div = document.createElement("div");
        div.classList.add("message-item");
        div.classList.add(msg.isRead === 0 ? "incoming" : "outgoing");

        const time = new Date(msg.timestamp).toLocaleString('en-IN', {
            timeZone: 'Asia/Kolkata',
            hour: '2-digit',
            minute: '2-digit',
            day: 'numeric',
            month: 'short'
        });

        let content = '';

        switch (msg.messageType) {
            case 'text':
                content = `<p>${msg.text}</p>`;
                break;

            case 'image':
                content = `
                    <img src="../api/download.php?id=${msg.mediaId}" alt="Image" class="message-media"/>
                    ${msg.text ? `<p>${msg.text}</p>` : ''}
                `;
                break;

            case 'video':
                content = `
                    <video controls class="message-media">
                        <source src="../api/download.php?id=${msg.mediaId}" type="${msg.mimeType}">
                        Your browser does not support the video tag.
                    </video>
                    ${msg.text ? `<p>${msg.text}</p>` : ''}
                `;
                break;

            case 'document':
                content = `
                    <a href="../api/download.php?id=${msg.mediaId}" target="_blank">
                        📄 ${msg.fileName || 'Download Document'}
                    </a>
                `;
                break;

            case 'contacts':
                content = `<p>📇 Contact shared</p>`;
                break;

            default:
                content = `<p>🔸 [${msg.messageType}] Not supported yet</p>`;
                break;
        }

        div.innerHTML = `
            <div class="bubble">
                ${content}
                <span class="timestamp">${time}</span>
            </div>
        `;
        container.appendChild(div);
    });

    // Auto scroll to bottom
    container.scrollTop = container.scrollHeight;
}
